using System;
using System.Drawing;
using System.Collections;
using System.Reflection;

using Microsoft.DirectX;

using DarkStrideToolbox;

namespace DarkStride.StellarLanes.SharedDLL
{
	public class Location
	{
		#region Properties
        private DSNetPKey m_oZoneID = null;
		private Vector2 m_vPos = Vector2.Empty;
		private Vector2 m_vVel = Vector2.Empty;
		private double m_nAngularMomentum = 0;
		private double m_nAngle = 0;

		private double m_nMaxVelocity = 1200;
		private double m_nMaxAngularVelocity = 4;
		#endregion

		public Location()
		{
		}
		public Location( Location oLocationToCopyFrom )
		{
			this.Pos = oLocationToCopyFrom.Pos;
			this.Vel = oLocationToCopyFrom.Vel;
			this.AngularMomentum = oLocationToCopyFrom.AngularMomentum;
			this.Angle = oLocationToCopyFrom.Angle;
			this.MaxVelocity = oLocationToCopyFrom.MaxVelocity;
			this.MaxAngularVelocity = oLocationToCopyFrom.MaxAngularVelocity;
            this.ZoneID = new DSNetPKey(oLocationToCopyFrom.ZoneID.HighNumber, oLocationToCopyFrom.ZoneID.LowNumber);
		}
        
        public void Advance( Session oSession, double nElapsedTime )
		{
			double nDecPerc = 0;

			//////////////////////////////////////////////////////////////////////////////////////////////////
			//Are we going to fast?
			//////////////////////////////////////////////////////////////////////////////////////////////////
			if( m_nMaxVelocity > 0 && this.Vel.Length() > m_nMaxVelocity )
			{
				//This is percentage over
				nDecPerc = ( this.Vel.Length() / m_nMaxVelocity ) - 1.0;
				//This is true precentage to decrease
				nDecPerc = nDecPerc / ( nDecPerc + 1 );
				//This is the decrease multiplier
				nDecPerc = 1 - nDecPerc;

				this.Vel = new Vector2( (float)( this.Vel.X * nDecPerc ),(float)( this.Vel.Y * nDecPerc ) );
			}
			if( m_nMaxAngularVelocity != 0 && this.AngularMomentum > m_nMaxAngularVelocity )
			{
				this.AngularMomentum = m_nMaxAngularVelocity;
			}
			else if( m_nMaxAngularVelocity != 0 && this.AngularMomentum < -m_nMaxAngularVelocity )
			{
				this.AngularMomentum = -m_nMaxAngularVelocity;
			}


			//////////////////////////////////////////////////////////////////////////////////////////////////
			//Advance us
			//////////////////////////////////////////////////////////////////////////////////////////////////
			this.Pos = new Vector2( this.Pos.X + this.Vel.X * (float)nElapsedTime,
									this.Pos.Y + this.Vel.Y * (float)nElapsedTime );
			this.Angle += this.AngularMomentum * nElapsedTime;
			this.Angle = DSMath.NormalizeRadAngle( this.Angle );


			//////////////////////////////////////////////////////////////////////////////////////////////////
			//Bounds checking on our world
			//////////////////////////////////////////////////////////////////////////////////////////////////
            if (this.Pos.X < -oSession.Width / 2.0 && this.Vel.X < 0)
			{
				this.Vel = new Vector2( this.Vel.X * -1,this.Vel.Y );
			}
            if (this.Pos.X > oSession.Width / 2.0 && this.Vel.X > 0)
			{
				this.Vel = new Vector2( this.Vel.X * -1,this.Vel.Y );
			}
            if (this.Pos.Y < -oSession.Height / 2.0 && this.Vel.Y < 0)
			{
				this.Vel = new Vector2( this.Vel.X,this.Vel.Y * -1 );
			}
            if (this.Pos.Y > oSession.Height / 2.0 && this.Vel.Y > 0)
			{
				this.Vel = new Vector2( this.Vel.X,this.Vel.Y * -1 );
			}
		}

		public virtual DSSerialize Serialize()
		{
			//string sRetVal = "";
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
			oSerialize.Set( 0,this.Pos.X );
			oSerialize.Set( 1,this.Pos.Y );
			oSerialize.Set( 2,this.Vel.X );
			oSerialize.Set( 3,this.Vel.Y );
			oSerialize.Set( 4,this.Angle );
			oSerialize.Set( 5,this.AngularMomentum );
			oSerialize.Set( 6,this.MaxVelocity );
			oSerialize.Set( 7,this.MaxAngularVelocity );
            oSerialize.Set(8, this.ZoneID.Serialize());
			

			return( oSerialize );
		}
		public virtual void DeSerialize( DSSerialize oSerialize )
		{
			float nTempX = 0, nTempY = 0;

			nTempX					= oSerialize.GetFloat( 0 );
			nTempY					= oSerialize.GetFloat( 1 );
			this.Pos				= new Vector2( nTempX,nTempY );
			nTempX					= oSerialize.GetFloat( 2 );
			nTempY					= oSerialize.GetFloat( 3 );
			this.Vel				= new Vector2( nTempX,nTempY );
			this.Angle				= oSerialize.GetDouble( 4 );
			this.AngularMomentum	= oSerialize.GetDouble( 5 );
			this.MaxVelocity		= oSerialize.GetDouble( 6 );
			this.MaxAngularVelocity	= oSerialize.GetDouble( 7 );

            this.ZoneID = new DSNetPKey(-1, -1);
            this.ZoneID.DeSerialize( (DSSerialize) oSerialize.Get(8) );
		}

		#region Properties
        public DSNetPKey ZoneID
        {
            get
            {
                return (m_oZoneID);
            }
            set
            {
                m_oZoneID = value;
            }
        }
		public Vector2 Pos
		{
			get
			{
				return( m_vPos );
			}
			set
			{
				m_vPos = value;
			}
		}
		public Vector2 Vel
		{
			get
			{
				return( m_vVel );
			}
			set
			{
if( value.X > 30000 || value.Y > 30000 )
{
    m_vVel = new Vector2(0, 0);
}
else
{
				m_vVel = value;
}
			}
		}

		public double AngularMomentum
		{
			get
			{
				return( m_nAngularMomentum );
			}
			set
			{
				m_nAngularMomentum = value;
			}
		}
		public double Angle
		{
			get
			{
				return( m_nAngle );
			}
			set
			{
				m_nAngle = value;
			}
		}

		public double MaxVelocity
		{
			get
			{
				return( m_nMaxVelocity );
			}
			set
			{
				m_nMaxVelocity = value;
			}
		}
		public double MaxAngularVelocity
		{
			get
			{
				return( m_nMaxAngularVelocity );
			}
			set
			{
				m_nMaxAngularVelocity = value;
			}
		}
		#endregion
	}	
	public class Structure
	{
		#region Properties
		private double m_nStructurePoints = 1;
		private double m_nMaxStructurePoints = 1;
		private double m_nPercentDamageRedux = 0;
		private double m_nFixedDamageRedux = 0;
		private double m_nMass = 1;
		#endregion

		public Structure()
		{
		}

		public virtual double ReduceDamage( double nDamageAmount )
		{
			double nTempDamage = nDamageAmount;

			nTempDamage -= nTempDamage * m_nPercentDamageRedux;
			nTempDamage -= m_nFixedDamageRedux;
			nTempDamage = Math.Max( 0,nTempDamage );

			return( nTempDamage );
		}
		public virtual double ReduceDamage( double nDamageAmount,double nFixedRedux,double nPercRedux )
		{
			double nTempDamage = nDamageAmount;

			nTempDamage -= nTempDamage * nPercRedux;
			nTempDamage -= nFixedRedux;
			nTempDamage = Math.Max( 0,nTempDamage );

			return( nTempDamage );
		}

		public virtual DSSerialize Serialize(Session oSession)
		{
			//string sRetVal = "";
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
			oSerialize.Set( 0,m_nStructurePoints );
			oSerialize.Set( 1,m_nMaxStructurePoints );
			oSerialize.Set( 2,m_nPercentDamageRedux );
			oSerialize.Set( 3,m_nFixedDamageRedux );
			oSerialize.Set( 4,m_nMass );
			

			return( oSerialize );
		}
        public virtual void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			m_nStructurePoints		= oSerialize.GetDouble( 0 );
			m_nMaxStructurePoints	= oSerialize.GetDouble( 1 );
			m_nPercentDamageRedux	= oSerialize.GetDouble( 2 );
			m_nFixedDamageRedux		= oSerialize.GetDouble( 3 );
			m_nMass					= oSerialize.GetDouble( 4 );
		}

		#region Properties
		public double StructurePoints
		{
			get
			{
				return( m_nStructurePoints );
			}
			set
			{
				m_nStructurePoints = value;
			}
		}
		public double MaxStructurePoints
		{
			get
			{
				return( m_nMaxStructurePoints );
			}
			set
			{
				m_nMaxStructurePoints = value;
			}
		}
		public double PercentDamageRedux
		{
			get
			{
				return( m_nPercentDamageRedux );
			}
			set
			{
				m_nPercentDamageRedux = value;
			}
		}
		public double FixedDamageRedux
		{
			get
			{
				return( m_nFixedDamageRedux );
			}
			set
			{
				m_nFixedDamageRedux = value;
			}
		}
		public virtual double Mass
		{
			get
			{
				return( m_nMass );
			}
			set
			{
				if( value == 0 )
				{
					m_nMass = value;
				}
				m_nMass = value;
			}
		}
		#endregion
	}

    public class Entitys : System.Collections.Generic.SortedList<string, Entity>
    {
        #region Member Variables
        //private  m_oEntitys
        #endregion

        public void Add(Session oSession, Entity oEntity, bool bNewEntityNeedsToBeResent)
        {
            this.Add(oEntity.PKey, oEntity);

            //Am I the host?  If not then we need to tell everyone else about this new entity
            if (bNewEntityNeedsToBeResent == true)
            {
                oSession.TransmitEntity(oEntity, true);
            }
        }
        public void Remove(Session oSession, Entity oDelEntity)
        {
            DSNetworkPacket oPacket = null;


            if (this.ContainsKey(oDelEntity.PKey) == true)
            {
                this.Remove(oDelEntity.PKey);
            }
            
            //Am I the host?  If not then we need to tell everyone else about this new entity
            if (Globals.Inst().IAmTheServer == true)
            {
                oPacket = NetMsg.Pckt_RemoveEntity(oDelEntity);
                oSession.SendMsgOnToAllPlayersInSession(oPacket);
            }
            else
            {
                NetMsg.Send_RemoveEntity(Globals.Inst().Network, Globals.Inst().Network.PlayerWhoIsHost, oDelEntity);
            }
        }

        #region Properties
        #endregion
    }
	public class Entity : Structure
	{
		#region Properties
        //Shrunk effect
        private double m_nTimeToShrinkFull = 0;
        private double m_nTimeToShrink = 0;
        private double m_nTimeToStayShrunk = 0;

		private double m_nTimeHasBeenDead = 0;
		private double m_nTimeSinceLastCollision = 0;

		private bool m_bRecoilsAfterCollision = true;
		private bool m_bCollidesWithOtherEntitys = true;

		private Location m_oLocation = new Location();
		private Location m_oServerLocation = null;
		private Location m_oLastServerLocation = null;

		private bool m_bFadingOut = false;
		private double m_nTimeFadeStarted = 0;
		private double m_nTimeToFadeOut = 0;

		//private string m_sGUID = "";
		private DSNetPKey m_oPKey = null;
		private double m_nTotalElapsedTime = 0;
		private bool m_bIsDead = false;
		private long m_nOwnerSocketID = MiscConstants.m_cNOSOCKETIDASSIGNED;
		private double m_nTimeSinceLastLiteSend = 0;
		private double m_nTimeSinceLastFullSend = 0;
		private bool m_bIsInPlay = true;
		private bool m_bIsVisible = true;

		private long m_nEXP = 0;
		private long m_nLevelID = 1;
		private long m_nNextLevelEXP = 0;
        private const double m_cLEVELUPGLARETIME = .8;
        private double m_nCountdownFromLevelUp = 0;

		private ArrayList m_oLevelSpecialTraits = new ArrayList();

		private long m_nEXPValue = 10;
		private enumEntityType m_nEntityType = enumEntityType.Ship;
		private bool m_bCanPickUpItems = false;
		private int m_nLootID = -1;

		//Orbiting data
		//private bool m_bIsOrbitingEntity = false;
		//private string m_sOrbitingEntityPKey = "";
		//private double m_nOrbitingDistance = 0;

		//Damage accuing and reporting
		private double m_nDamageAccrued = 0;
		private double m_nDamageDone = 0;
		private double m_nAppTimeDamageStartedAccruing = 0;
		private double m_nAppTimeOfLastDamageAccruing = 0;
		private double m_nAppTimeOfLastDamageRendering = 0;
		private Vector2 m_vDamageAcrruedLocation = Vector2.Empty;
		private bool m_bDamageHasBeenAccrued = false;

		//Sometimes ships can pick up temporary properties, this is where we track it
		private double[] m_naCEPropertyTotals = new double[ (int)enumEntProperties.LastItem ];
		private double[] m_daCEPropertyTotalsLastUpdate = new double[ (int)enumEntProperties.LastItem ];
		private ArrayList m_oTempProperties = new ArrayList();

		public bool m_bIsWaveEntity = false;
		#endregion


		public Entity()
		{
			//m_sGUID = DSMisc.GetGUID();
			m_oPKey = Globals.Inst().PrimaryKey.GetNewPrimarykey();
		}

		public virtual void Advance( Session oSession,double nElapsedTime )
		{
			TemporaryPropertyChange oTempProp = null;
			string sMsg = string.Empty;
			double nLinearDrag = 0;
			double nAngularDrag = 0;
			double nAngle = 0;
			double nTurning = 0;
			double nThrust = 0;
			double nVelLen = 0;


            //we have some kewl special effects we do if we just leveled up
            if (m_nCountdownFromLevelUp > 0)
            {
                m_nCountdownFromLevelUp -= nElapsedTime;
                if (m_nCountdownFromLevelUp < 0)
                {
                    m_nCountdownFromLevelUp = 0;
                }
            }

			//Verify the mass isn't zero
			if( this.Mass == 0 )
			{
				throw new System.Exception( "The mass of this entity is zero." );
			}

			//Update our timing
			m_nTimeSinceLastLiteSend += nElapsedTime;
			//m_nTimeSinceLastFullSend += nElapsedTime;
            if (m_nTimeToShrink > 0)
            {
                m_nTimeToShrink = DSMisc.Max(0, m_nTimeToShrink - nElapsedTime);
            }
            else if( m_nTimeToStayShrunk > 0 )
            {
                m_nTimeToStayShrunk = DSMisc.Max(0, m_nTimeToStayShrunk - nElapsedTime);
            }
			m_nTotalElapsedTime += nElapsedTime;
			m_nTimeSinceLastCollision += nElapsedTime;

			//If i'm dead then I pop-out after a time
			if( this.IsDead == true )
			{
				m_nTimeHasBeenDead += nElapsedTime;
			}

			//Advance our temporary properties
			for( int i=0 ; i<m_oTempProperties.Count ; i++ )
			{
				if( i > m_oTempProperties.Count ){ break; }

				oTempProp = (TemporaryPropertyChange)m_oTempProperties[i];
				if( Globals.Inst().GameEngine.AppTime - oTempProp.StartTime > oTempProp.Duration )
				{
					m_oTempProperties.RemoveAt( i );
				}
			}

			//Check for any nearby gravitational bodies
			AdvanceGravitationalPull( oSession,nElapsedTime );

			//Finally advance our location
            if (oSession.Debug_Pause == false)
			{
				//Advance our movement
				nThrust = GetPropertyTotal( enumEntProperties.Movement_LinearThrust ) * nElapsedTime;
				nTurning = ( GetPropertyTotal( enumEntProperties.Movement_AngularThrust ) / this.Mass );// * nElapsedTime;
				this.Vel = new Vector2( 
						(float)( this.Vel.X - Math.Cos( this.Angle + Math.PI / 2.0f ) * nThrust ),
						(float)( this.Vel.Y - Math.Sin( this.Angle + Math.PI / 2.0f ) * nThrust ) );
				this.AngularMomentum = nTurning;
				if( this.ServerLocation != null )
				{
					this.ServerLocation.AngularMomentum = nTurning;
					this.ServerLocation.Vel = new Vector2( 
						(float)( this.ServerLocation.Vel.X - Math.Cos( this.ServerLocation.Angle ) * nThrust ),
						(float)( this.ServerLocation.Vel.Y - Math.Sin( this.ServerLocation.Angle ) * nThrust ) );
				}

				//Apply our drag
				nLinearDrag = GetPropertyTotal( enumEntProperties.Movement_EtherialLinearDrag );
				nAngularDrag = GetPropertyTotal( enumEntProperties.Movement_EtherialAngularDrag );
				//Reduce our vel by drag
				nAngle = DSMath.CalculateRadAngle( 0,0,this.Vel.X,this.Vel.Y );
				nVelLen = DSMath.MoveTowords( 0,this.Vel.Length(),nLinearDrag*nElapsedTime );
				this.Vel = new Vector2( (float)( Math.Cos( nAngle ) * nVelLen ),
										(float)( Math.Sin( nAngle ) * nVelLen ) );
					//(float)DSMath.MoveTowords( 0,this.Vel.X,nLinearDrag*nElapsedTime ),
					//(float)DSMath.MoveTowords( 0,this.Vel.Y,nLinearDrag*nElapsedTime ) );				
				this.AngularMomentum = DSMath.MoveTowords( 0,this.AngularMomentum,nAngularDrag*nElapsedTime );
				if( this.ServerLocation != null )
				{
					//Reduce our vel by drag
					nAngle = DSMath.CalculateRadAngle( 0,0,this.ServerLocation.Vel.X,this.ServerLocation.Vel.Y );
					nVelLen = DSMath.MoveTowords( 0,this.ServerLocation.Vel.Length(),nLinearDrag*nElapsedTime );
					this.ServerLocation.Vel = new Vector2(	(float)( Math.Cos( nAngle ) * nVelLen ),
															(float)( Math.Sin( nAngle ) * nVelLen ) );
					/*this.ServerLocation.Vel = new Vector2( 
						(float)DSMath.MoveTowords( 0,this.ServerLocation.Vel.X,nLinearDrag*nElapsedTime ),
						(float)DSMath.MoveTowords( 0,this.ServerLocation.Vel.Y,nLinearDrag*nElapsedTime ) );*/
					this.ServerLocation.AngularMomentum = DSMath.MoveTowords( 0,this.ServerLocation.AngularMomentum,nAngularDrag*nElapsedTime );
				}

				//Do the basics
                m_oLocation.Advance(oSession, nElapsedTime);
				if( m_oServerLocation != null )
				{
					m_oServerLocation.Advance( oSession,nElapsedTime );
				}
			}

			//Check our fadeout
			if( m_bFadingOut == true && this.TotalElapsedTime - m_nTimeFadeStarted >= m_nTimeToFadeOut )
			{
				oSession.RemoveEntity( this );
			}

			#region Dead Reackoning
			/*if( m_oServerLocation != null && this.PKey != Globals.Inst().OurShip.PKey )
			{
				m_oLocation.Angle = DSMath.NormalizeRadAngle( m_oLocation.Angle );
				m_oServerLocation.Angle = DSMath.NormalizeRadAngle( m_oServerLocation.Angle );

				double nAngularChange = 3;
				double nOldAngle = m_oLocation.Angle;
				double nDist = DSMath.AngleRadDiff( m_oLocation.Angle,m_oServerLocation.Angle );
				if( DSMath.AngleRadDiff( m_oLocation.Angle,m_oServerLocation.Angle ) < .1 )
				{
					m_oLocation.AngularMomentum = m_oServerLocation.AngularMomentum;
					m_oLocation.Angle = m_oServerLocation.Angle;
				}
				else if( m_oServerLocation.AngularMomentum == 0 )
				{
					if( DSMath.ShortestDirectionToRadAngle( m_oLocation.Angle,m_oServerLocation.Angle ) ==
						DSMath.enumSHORTDISTTOANGLE.enumCW )
					{
						m_oLocation.AngularMomentum = -nAngularChange;
					}
					else if( DSMath.ShortestDirectionToRadAngle( m_oLocation.Angle,m_oServerLocation.Angle ) ==
						DSMath.enumSHORTDISTTOANGLE.enumCCW )
					{
						m_oLocation.AngularMomentum = nAngularChange;
					}					
				}

				  
				 
				float nNewX = 0;
				float nNewY = 0;
				if( DSMath.Distance( m_oLocation.Pos.X,m_oLocation.Pos.Y,m_oServerLocation.Pos.X,m_oServerLocation.Pos.Y ) > 500 )
				{
					m_oLocation.Pos = new Vector2( m_oServerLocation.Pos.X,m_oServerLocation.Pos.Y );
					nNewX = m_oServerLocation.Vel.X;
					nNewY = m_oServerLocation.Vel.Y;
				}
				else
				{
					if( Math.Abs( m_oLocation.Pos.X - m_oServerLocation.Pos.X ) < 5 )
					{
						nNewX = m_oServerLocation.Vel.X;
						m_oLocation.Pos = new Vector2( m_oServerLocation.Pos.X,m_oLocation.Pos.Y );
					}
					else 
					{
						float nAdjust = ( (float)this.MaxVelocity / 10f );
						//float nAdjust = 10;
						nNewX = m_oLocation.Vel.X - nAdjust * (float)DSMath.Sign( m_oLocation.Pos.X - m_oServerLocation.Pos.X );
					}
					if( Math.Abs( m_oLocation.Pos.Y - m_oServerLocation.Pos.Y ) < 5 )
					{
						nNewY = m_oServerLocation.Vel.Y;
						m_oLocation.Pos = new Vector2( m_oLocation.Pos.X,m_oServerLocation.Pos.Y );
					}
					else 
					{
						float nAdjust = ( (float)this.MaxVelocity / 10f );
						//float nAdjust = 10;
						nNewY = m_oLocation.Vel.Y - nAdjust * (float)DSMath.Sign( m_oLocation.Pos.Y - m_oServerLocation.Pos.Y );
					}
				}
				m_oLocation.Vel = new Vector2( nNewX,nNewY );
			}*/
			#endregion

			#region Update our damage accruing
			double nAppTime = Globals.Inst().GameEngine.AppTime;
			//Lets try imediantly reporting it, the whole explosion thing is turned off for now.
			if( m_bDamageHasBeenAccrued == true && //m_nDamageAccrued > 0 && 
				(
					nAppTime - m_nAppTimeDamageStartedAccruing > 1 ||
					nAppTime - m_nAppTimeOfLastDamageAccruing > .25 ||
					nAppTime - m_nAppTimeOfLastDamageRendering > 2 
				)
			  )
			{
				sMsg = "-" + m_nDamageAccrued.ToString( "0" );
				if( m_nDamageDone != m_nDamageAccrued && m_nDamageDone > 0 )
				{
					sMsg += " (" + m_nDamageDone.ToString( "0" ) + ")";
				}

				if( Globals.Inst().IAmTheServer == true )
				{
					//m_nDamageDone
					NetMsg.Send_RollingText( Globals.Inst().GameEngine.DirectPlay,
						m_vDamageAcrruedLocation,this.Vel,System.Drawing.Color.Red.ToArgb(),sMsg );
				}

                oSession.AddRollingText(m_vDamageAcrruedLocation, this.Vel, sMsg, System.Drawing.Color.Red);

				m_nAppTimeOfLastDamageRendering = Globals.Inst().GameEngine.AppTime;
				m_nDamageAccrued = 0;
				m_bDamageHasBeenAccrued = false;
				m_nAppTimeDamageStartedAccruing = 0;
			}
			#endregion
		}
		public virtual void AdvanceGravitationalPull( Session oSession,double nElapsedTime )
		{
            Zone oMyZone = null;
			Vector2 vGravPull = Vector2.Empty;


            if (this.SuperMass == false && this.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED )
			{
                oMyZone = oSession.Zones[this.Location.ZoneID.ToString()];

                vGravPull = oMyZone.GetGravitationalPull(nElapsedTime, this.Pos);
				if( vGravPull.Length() > 20 )
				{
					this.Vel = new Vector2( (float)( this.Vel.X - vGravPull.X / this.Mass ),
							                (float)( this.Vel.Y - vGravPull.Y / this.Mass ) );
				}
			}
		}
		public virtual void PreRenderRenderSettingsChange( RenderSettings oRenderSettings )
		{
            double nPerc = 0;

            if (m_nTimeToShrink > 0 || m_nTimeToStayShrunk > 0)
            {
                if (m_nTimeToShrink > 0 )
                {
                    nPerc = m_nTimeToShrink / m_nTimeToShrinkFull;
                }
                else
                {
                    nPerc = 0;
                }
                oRenderSettings.ZoomLevel *= nPerc;
                oRenderSettings.BaseDrawColor = System.Drawing.Color.FromArgb(
                                    (int)(oRenderSettings.BaseDrawColor.R * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.G * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.B * nPerc));
            }
		}
		public virtual void Render( RenderSettings oRenderSettings )
		{
            double nPerc = 0;


            if (m_nCountdownFromLevelUp > 0)
            {
                nPerc = m_nCountdownFromLevelUp / m_cLEVELUPGLARETIME;

                //Go from white to black
                if (nPerc > .5)
                {
                    nPerc = ( nPerc - .5 ) / .5;
                    oRenderSettings.BaseDrawColor = System.Drawing.Color.FromArgb(
                                    (int)(oRenderSettings.BaseDrawColor.R * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.G * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.B * nPerc));
                }
                //Go from black back to white
                else
                {
                    nPerc = 1 - ( nPerc / .5 );
                    oRenderSettings.BaseDrawColor = System.Drawing.Color.FromArgb(
                                    (int)(oRenderSettings.BaseDrawColor.R * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.G * nPerc),
                                    (int)(oRenderSettings.BaseDrawColor.B * nPerc));
                }
            }
		}
		public virtual Vector2 GetRotatedWorldPt( Vector2 vWorldPt )
		{
			Vector2 vWorldPtEndPt = Vector2.Empty;
			double nAngle = 0;
			double nDist = 0;

			
			//10/08/2007 Chris Hill  Ok this may seem odd, but its right.  I just changed it so that - is up, to 
			//be in line with the graphic system of 0,0 in the upper left corner.  Problem is the angle system
			//(aka radians) uses -Y to be down.  So we have to flip it for all angle calculations.
			nAngle = DSMath.CalculateRadAngle( this.Pos.X,-this.Pos.Y,vWorldPt.X,-vWorldPt.Y );
			nDist = DSMath.Distance( this.Pos.X,this.Pos.Y,vWorldPt.X,vWorldPt.Y );

			vWorldPtEndPt = new Vector2( 
				(float)( Math.Cos( nAngle - this.Angle ) * nDist + this.Pos.X ),
				(float)( Math.Sin( nAngle - this.Angle ) * -nDist + this.Pos.Y ) );


			return( vWorldPtEndPt );
		}
		public void RenderSimple( RenderSettings oRenderSettings,string sTextureKey,Vector2 vSize )
		{
			Vector2 vScreenPt = Vector2.Empty;
			Vector2 vWorldUpperLeftCorner = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;
			double nPercTrans = 0;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
                vWorldUpperLeftCorner = new Vector2((float)(this.Pos.X - vSize.X / 2.0),
                                                    (float)(this.Pos.Y - vSize.Y / 2.0));
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldUpperLeftCorner );

				oRenderRect = new System.Drawing.Rectangle( 
					(int)vScreenPt.X,(int)vScreenPt.Y,
                    (int)(vSize.X * oRenderSettings.ZoomLevel),
                    (int)(vSize.Y * oRenderSettings.ZoomLevel));

				nPercTrans = DSMisc.Max( oRenderSettings.PercentTransparent,this.FadeoutTransparency() );

                Globals.Inst().GameEngine.RenderTexture2D(sTextureKey,
					System.Drawing.Rectangle.Empty,oRenderRect,
					new Vector2( oRenderRect.Width / 2.0f,oRenderRect.Height / 2.0f ),this.Angle,
					nPercTrans,false,oRenderSettings.BaseDrawColor.ToArgb() );
			}
        }
        public void LevelUp()
        {
            m_nCountdownFromLevelUp = m_cLEVELUPGLARETIME;
        }

		public virtual ArrayList GetRegions()
		{
			return( new ArrayList() );
		}
		public virtual bool Collision( Session oSession,Entity oCollidedWith,double nElapsedTime,RegionPoint oCollisionRegion )
		{
			return( true );
		}
		public void ApplyThrust( Vector2 vOffsetFromCenterOfEntitysMass,double nAngle,double nThrustAmount )
		{
			double nThrust = nThrustAmount / this.Mass;
			Vector2 vOffset = vOffsetFromCenterOfEntitysMass;
			//vOffset.Normalize();
			double nAngleToCOM = DSMath.CalculateRadAngle( 0,0,vOffset.X,vOffset.Y ) - this.Angle;
			nAngleToCOM = DSMath.NormalizeRadAngle( nAngleToCOM );
			

			//Thrust to apply to linear motion
			double nLinearThrust = Math.Cos( nAngle + nAngleToCOM + Math.PI / 2.0 ) * nThrust;
			this.Vel = new Vector2( 
				(float)( this.Vel.X - Math.Cos( nAngleToCOM ) * nLinearThrust ),
				(float)( this.Vel.Y - Math.Sin( nAngleToCOM ) * nLinearThrust ) );

			//Thrust to apply to angular motion
			double nAngularThrust = Math.Sin( nAngle + nAngleToCOM + Math.PI / 2.0 ) * nThrust * .1;
			//If the angular change is against the rotation it produces linear velocity
			double nAngleChange = nAngularThrust / vOffsetFromCenterOfEntitysMass.Length();
			if( vOffsetFromCenterOfEntitysMass.Length() == 0 )
			{
				nAngleChange = 0;
			}
			if( DSMath.Sign( nAngleChange ) == DSMath.Sign( this.AngularMomentum ) &&
				nAngleChange != 0 && this.AngularMomentum != 0 )
			{
				double nPercUsedForAngular = DSMisc.Min( 1,Math.Abs( this.AngularMomentum ) / Math.Abs( nAngleChange ) );
				this.Vel = new Vector2( 
					(float)( this.Vel.X - Math.Cos( nAngleToCOM+Math.PI/2.0 ) * -nAngularThrust * 10 * nPercUsedForAngular ),
					(float)( this.Vel.Y - Math.Sin( nAngleToCOM+Math.PI/2.0 ) * -nAngularThrust * 10 * nPercUsedForAngular ) );
			}
			this.AngularMomentum -= nAngleChange;
		}
        public void StartShrinkEeffect(double nTimeToShrink,double nTimeToStayShrunk)
        {
            m_nTimeToShrinkFull = nTimeToShrink;
            m_nTimeToShrink = nTimeToShrink;
            m_nTimeToStayShrunk = nTimeToStayShrunk;
        }

		//Returns the amount of damage not absorbed by the entity
        public virtual double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
			double nDamageLeft = 0;
			double nReducedDamage = 0;


			nReducedDamage = this.ReduceDamage( nAmount );

			//Mark the damage in the session
			this.AccrueDamageRollingText( oDamageWhere.ConvertToPointLocation(),nReducedDamage,nAmount );
			//How much of the damage are we going to use?
			nDamageLeft = nAmount - DSMisc.Min( nReducedDamage,this.StructurePoints );
			this.StructurePoints -= nReducedDamage;

			if( this.StructurePoints < 0 && this.IsDead == false )
			{
				this.IsDead = true;
                HandleEntityDeath(oSession, oEntityDealingDamage);
			  }


			return( nDamageLeft );
		}
        public virtual void HandleEntityDeath(Session oSession,Entity oEntityDealingDamage)
        {
            if (Globals.Inst().IAmTheServer == true && this.EXPValue > 0)
            {
                //We give the exp to every ship this player controls, that can pick up items and is in this zone
                foreach (Entity oLoopEntity in oSession.GetEntitysInMyZone(this).Values)
                {
                    if (oLoopEntity.OwnerSocketID == oEntityDealingDamage.OwnerSocketID && oLoopEntity.CanPickUpItems == true)
                    {
                        NetMsg.Send_EntityReceivesEXP(Globals.Inst().GameEngine.DirectPlay, oLoopEntity, this.EXPValue);
                        oSession.GiveExpToEntity(oLoopEntity, this.EXPValue);
                    }
                }
            }

            //Also there is a small chance of loot
            if (Globals.Inst().IAmTheServer == true && this.LootID > 0 &&
                oEntityDealingDamage.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED)
            {
                oSession.CreateFloatingLoot(this.MostUpToDateLoc.ZoneID.ToString(), this.Pos, this.LootID);
            }

            oSession.TransmitEntity(this, true);
        }
		public virtual Region HitTest( Region oTestRegion )
		{
			Region oRegion = null;
			Region oHitTestRegion = null;
			Region oRetVal = null;
			ArrayList oRegions = GetRegions();			

			for( int i=0 ; i<oRegions.Count ; i++ )
			{
				oRegion = (Region)oRegions[i];
				oHitTestRegion = oRegion.HitTest( oTestRegion );
				if( Region.IsEmpty( oHitTestRegion ) == false )
				{
					oRetVal = oHitTestRegion;
					break;
				}
			}
			return( oRetVal );
		}
        public virtual void KeyProcess( Session oSession,Microsoft.DirectX.DirectInput.Key oKey, bool bPressed)
		{
		}
		public void AccrueDamageRollingText( Vector2 vDamagePos,double nDamageToAccrue,double nDamageDone )
		{
			double nAppTime = Globals.Inst().GameEngine.AppTime;


			if( nDamageToAccrue > 0 || nDamageDone > 0 )
			{
				if( m_bDamageHasBeenAccrued == false || //m_nDamageAccrued == 0 ||
					(
						vDamagePos != Vector2.Empty &&
						DSMisc.Distance( vDamagePos.X,vDamagePos.Y,
						m_vDamageAcrruedLocation.X,m_vDamageAcrruedLocation.Y ) > 50 
					)
				  )
				{
					m_nDamageAccrued = nDamageToAccrue;
					m_nDamageDone = nDamageDone;
					m_vDamageAcrruedLocation = vDamagePos;
					m_nAppTimeDamageStartedAccruing = nAppTime;
					m_nAppTimeOfLastDamageAccruing = nAppTime;
				}
				else
				{
					m_nAppTimeOfLastDamageAccruing = nAppTime;
					m_nDamageAccrued += nDamageToAccrue;
					m_nDamageDone += nDamageDone;
				}

				m_bDamageHasBeenAccrued = true;
			}
		}
		public void StartFadeout( double nTimeToFadeOut )
		{
			if( m_bFadingOut == false )
			{
				m_bFadingOut = true;
				m_nTimeFadeStarted = this.TotalElapsedTime;
				m_nTimeToFadeOut = nTimeToFadeOut;
			}
		}
		public double FadeoutTransparency()
		{
			double nPercTrans = 0;

			if( m_bFadingOut == true )
			{
				nPercTrans = ( this.TotalElapsedTime - m_nTimeFadeStarted ) / m_nTimeToFadeOut;
			}
			else
			{
				nPercTrans = 0;
			}

			return( nPercTrans );
		}

		public virtual double DelayBetweenLiteSends()
		{
			return( 1 );
		}
		public virtual double DelayBetweenFullSends()
		{
			return( 10 );
		}

		public virtual void NetworkMessageReceived( Session oSession,long nMessageType,string sMessage )
		{
		}

		public override DSSerialize Serialize(Session oSession)
		{
			DSSerialize oSerialize = new DSSerialize();


			//Now save our data
			oSerialize.Set( 0,m_oLocation.Serialize() );
			oSerialize.Set( 1,m_oPKey.Serialize() );
			oSerialize.Set( 2,m_nTotalElapsedTime );
			oSerialize.Set( 3,this.IsDead );
			oSerialize.Set( 4,base.Serialize(oSession) );
			oSerialize.Set( 5,m_nOwnerSocketID );
			oSerialize.Set( 6,m_bRecoilsAfterCollision );
			oSerialize.Set( 7,m_bCollidesWithOtherEntitys );
			oSerialize.Set( 8,m_nEXP );
			oSerialize.Set( 9,Convert.ToInt64( m_nEntityType ) );
			//oSerialize.Set( 10,m_bSuperMass );
			/*oSerialize.Set( 11,m_bIsOrbitingEntity );
			if( m_bIsOrbitingEntity == true )
			{
				oSerialize.Set( 12,m_sOrbitingEntityPKey );
				oSerialize.Set( 13,m_nOrbitingDistance );
			}*/
			oSerialize.Set( 11,m_bCanPickUpItems );
			oSerialize.Set( 12,m_bIsWaveEntity );
			

			return( oSerialize );
		}
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
		{
			m_oLocation.DeSerialize( (DSSerialize)oSerialize.Get( 0 ) );
			m_oPKey.DeSerialize( (DSSerialize)oSerialize.Get( 1 ) );
			m_nTotalElapsedTime				= oSerialize.GetDouble( 2 );
			this.IsDead						= oSerialize.GetBool( 3 );
			base.DeSerialize( oSession,(DSSerialize)oSerialize.Get( 4 ) );
			m_nOwnerSocketID				= oSerialize.GetLong( 5 );
			m_bRecoilsAfterCollision		= oSerialize.GetBool( 6 );
			m_bCollidesWithOtherEntitys		= oSerialize.GetBool( 7 );
			//The property does some stuff.  So has to be this.EXP.
			this.EXP							= oSerialize.GetLong( 8 );
			m_nEntityType					= (enumEntityType)oSerialize.GetLong( 9 );
			//m_bSuperMass					= oSerialize.GetBool( 10 );
			//m_bIsOrbitingEntity				= oSerialize.GetBool( 11 );
			//m_sOrbitingEntityPKey			= oSerialize.GetString( 12 );
			//m_nOrbitingDistance				= oSerialize.GetDouble( 13 );
			m_bCanPickUpItems				= oSerialize.GetBool( 11 );
			m_bIsWaveEntity					= oSerialize.GetBool( 12 );
		}

		#region Property Management
		public void AddTemporaryProperty( enumEntProperties nProperty, double nChange, double nDuration )
		{
			TemporaryPropertyChange oNewProp = null;


			oNewProp = new TemporaryPropertyChange();
			oNewProp.StartTime = Globals.Inst().GameEngine.AppTime;
			oNewProp.Property = nProperty;
			oNewProp.Change = nChange;
			oNewProp.Duration = nDuration;

			m_oTempProperties.Add( oNewProp );
		}

		public double GetPropertyTotal( enumEntProperties nPropertyToGet )
		{
			double nPropertyTotal = 0;
			//double nElapsed = 0;

			//A bug in this code was causing it to always re-calculate... which is good because the freq.
			//timer can't go low enough to handle the rapid changes in game actions.  So for now just always
			//recalculate it.

			//nElapsed = Globals.Inst().GameEngine.AppTime - m_naCEPropertyTotals[ (int)nPropertyToGet ];
			//nElapsed = DSMisc.GetQueryPerformanceCounter() - m_naCEPropertyTotals[ (int)nPropertyToGet ];
			//nElapsed = nElapsed / DSMisc.GetQueryPerformanceFrequency();

			//If we haven't updated this number yet or its been more than one second since we did update it,
			//go through and re-calculate the values.
			//if( m_daCEPropertyTotalsLastUpdate[ (int)nPropertyToGet ] == 0 || nElapsed > .10 )
			{
				nPropertyTotal = CalculatePropertyTotal( nPropertyToGet );
				m_naCEPropertyTotals[ (int)nPropertyToGet ] = nPropertyTotal;
				m_daCEPropertyTotalsLastUpdate[ (int)nPropertyToGet ] = Globals.Inst().GameEngine.AppTime;
			}
			/*else
			{
				nPropertyTotal = m_naCEPropertyTotals[ (int)nPropertyToGet ];
			}*/


			return( nPropertyTotal );
		}
		public virtual double CalculatePropertyTotal( enumEntProperties nPropertyToGet )
		{
			TemporaryPropertyChange oTempProp = null;
			double nPropertyTotal = 0;


			//Do the basics
			if( nPropertyToGet == enumEntProperties.Structure_Mass )
			{
				nPropertyTotal += base.Mass;
			}
			
			
			//Check our temporary properties
			for( int i=0 ; i<m_oTempProperties.Count ; i++ )
			{
				if( i > m_oTempProperties.Count ){ break; }

				oTempProp = (TemporaryPropertyChange)m_oTempProperties[i];
				if( oTempProp.Property == nPropertyToGet )
				{
					nPropertyTotal += oTempProp.Change;
				}
			}
			


			return( nPropertyTotal );
		}

		#endregion
		#region Companion Classes
		private class TemporaryPropertyChange
		{
			#region Member Variables
			private enumEntProperties m_nProperty = enumEntProperties.LastItem;
			private double m_nStartTime = 0;
			private double m_nDuration = 0;
			private double m_nChange = 0;
			#endregion

			#region Properties
			public enumEntProperties Property 
			{
				get
				{
					return( m_nProperty );
				}
				set
				{
					m_nProperty = value;
				}
			}
			public double StartTime
			{
				get
				{
					return( m_nStartTime );
				}
				set
				{
					m_nStartTime = value;
				}
			}
			public double Duration
			{
				get
				{
					return( m_nDuration );
				}
				set
				{
					m_nDuration = value;
				}
			}
			public double Change
			{
				get
				{
					return( m_nChange );
				}
				set
				{
					m_nChange = value;
				}
			}
			#endregion
		}

		#endregion
       

		#region Properties
		public Vector2 Pos
		{
			get
			{
				return( m_oLocation.Pos );
			}
			set
			{
				m_oLocation.Pos = value;
			}
		}
		public Vector2 Vel
		{
			get
			{
				return( m_oLocation.Vel );				
			}
			set
			{
				m_oLocation.Vel = value;
			}
		}
		public double AngularMomentum
		{
			get
			{
				return( m_oLocation.AngularMomentum );
			}
			set
			{
				m_oLocation.AngularMomentum = value;
			}
		}
		public double Angle
		{
			get
			{
				return( m_oLocation.Angle );				
			}
			set
			{
				m_oLocation.Angle = DSMath.NormalizeRadAngle( value );
			}
		}
		/*public string GUID
		{
			get
			{
				return( m_sGUID );
			}
		}*/
		public string PKey
		{
			get
			{
				return( m_oPKey.ToString() );
			}
		}
		public DSNetPKey PKeyObject
		{
			set
			{
				m_oPKey = value;
			}
			get
			{
				return( m_oPKey );
			}
		}
		public double TotalElapsedTime
		{
			get
			{
				return( m_nTotalElapsedTime );
			}
			set
			{
				m_nTotalElapsedTime = value;
			}
		}
		public bool IsDead
		{
			get
			{
				return( m_bIsDead );
			}
			set
			{
                if (m_bIsDead == true && value == false)
                {
                    m_bIsDead = value;
                }
				m_bIsDead = value;
			}
		}
		public long OwnerSocketID
		{
			get
			{
				return( m_nOwnerSocketID );
			}
			set
			{
				m_nOwnerSocketID = value;
			}
		}
		public double TimeSinceLastLiteSend
		{
			get
			{
				return( m_nTimeSinceLastLiteSend );
			}
			set
			{
				m_nTimeSinceLastLiteSend = value;
			}
		}
		public double TimeSinceLastFullSend
		{
			get
			{
				return( m_nTimeSinceLastFullSend );
			}
			set
			{
				m_nTimeSinceLastFullSend = value;
			}
		}
		public Location Location
		{
			get
			{
				return( m_oLocation );
			}
			set
			{
				m_oLocation = value;
			}
		}
		public Location ServerLocation
		{
			get
			{
				return( m_oServerLocation );
			}
			set
			{
				m_oServerLocation = value;
			}
		}
		public Location LastServerLocation
		{
			get
			{
				return( m_oLastServerLocation );
			}
			set
			{
				m_oLastServerLocation = value;
			}
		}
		public double TimeHasBeenDead
		{
			get
			{
				return( m_nTimeHasBeenDead );
			}
			set
			{
				m_nTimeHasBeenDead = value;
			}
		}
		public double TimeSinceLastCollision
		{
			get
			{
				return( m_nTimeSinceLastCollision );
			}
			set
			{
				m_nTimeSinceLastCollision = value;
			}
		}

		public bool CollidesWithOtherEntitys
		{
			get
			{
				return( m_bCollidesWithOtherEntitys );
			}
			set
			{
				m_bCollidesWithOtherEntitys = value;
			}
		}
		public bool RecoilsAfterCollision
		{
			get
			{
				return( m_bRecoilsAfterCollision );
			}
			set
			{
				m_bRecoilsAfterCollision = value;
			}
		}

		public bool IsInPlay
		{
			get
			{
				return( m_bIsInPlay );
			}
			set
			{
				m_bIsInPlay = value;
			}
		}

		public Location MostUpToDateLoc
		{
			get
			{
				if( this.ServerLocation == null )
				{
					return( this.Location );
				}
				else
				{
					return( this.ServerLocation );
				}
			}
		}
		public virtual double MaxVelocity
		{
			get
			{
				return( this.MostUpToDateLoc.MaxVelocity );
			}
			set
			{
				this.Location.MaxVelocity = value;
				if( this.ServerLocation != null )
				{
					this.ServerLocation.MaxVelocity = value;
				}
			}
		}
		public double MaxAngularVelocity
		{
			get
			{
				return( this.MostUpToDateLoc.MaxAngularVelocity );
			}
			set
			{
				this.Location.MaxAngularVelocity = value;
				if( this.ServerLocation != null )
				{
					this.ServerLocation.MaxAngularVelocity = value;
				}
			}
		}
		public bool IsVisible
		{
			get
			{
				return( m_bIsVisible );
			}
			set
			{
				m_bIsVisible = value;
			}
		}

		public long EXP
		{
			get
			{
				return( m_nEXP );
			}
			set
			{
				Session.GetLevelInfo( value,ref m_nLevelID,ref m_nNextLevelEXP,ref m_oLevelSpecialTraits );
				m_nEXP = value;
			}
		}
		public long LevelID
		{
			get
			{
				return( m_nLevelID );
			}
		}
		public ArrayList LevelSpecialTraits
		{
			get
			{
				return( m_oLevelSpecialTraits );
			}
		}

		public long NextLevelEXP
		{
			get
			{
				return( m_nNextLevelEXP );
			}
		}

		public virtual long EXPValue
		{
			get
			{
				return( m_nEXPValue );
			}
			set
			{
				m_nEXPValue = value;
			}
		}
		public enumEntityType EntityType
		{
			get
			{
				return( m_nEntityType );
			}
			set
			{
				m_nEntityType = value;
			}
		}
		public bool SuperMass
		{
			get
			{
                bool bSuperMass = false;
                if (this.GetType() == typeof(SuperMass) || this.GetType().BaseType == typeof( SuperMass))
                {
                    bSuperMass = true;
                }
                return (bSuperMass);
                //return (false);
			}
		}
		public bool CanPickUpItems
		{
			get
			{
				return( m_bCanPickUpItems );
			}
			set
			{
				m_bCanPickUpItems = value;
			}
		}		
		public int LootID
		{
			get
			{
				return( m_nLootID );
			}
			set
			{
				m_nLootID = value;
			}
		}
		public virtual string Name
		{
			get
			{
				return( string.Empty );
			}
		}
		public bool IsWaveEntity
		{
			get
			{
				return( m_bIsWaveEntity );
			}
			set
			{
				m_bIsWaveEntity = value;
			}
		}
		#endregion
	}
    public class FloatingLoot : Entity
    {
        #region Properties
        private const double m_cDRAGSPEED = .1;
        private Item m_oItemToBeReceived = null;
        private Vector2 m_vSize = new Vector2(20, 20);
        #endregion


        public FloatingLoot()
        {
            double nVelDrift = 25;

            this.EntityType = enumEntityType.Loot;
            this.AngularMomentum = DSMisc.GetRnd() * 1 - .5;
            this.RecoilsAfterCollision = false;
            this.Vel = new Vector2((float)(DSMisc.GetRnd() * nVelDrift * 2 - nVelDrift),
                                    (float)(DSMisc.GetRnd() * nVelDrift * 2 - nVelDrift));
        }

        public override void Advance(Session oSession, double nElapsedTime)
        {
            base.Advance(oSession, nElapsedTime);

            //Slow us down... don't want loot drifting to far.
            this.Vel = new Vector2((float)(this.Vel.X * (1 - m_cDRAGSPEED * nElapsedTime)),
                                    (float)(this.Vel.Y * (1 - m_cDRAGSPEED * nElapsedTime)));

            if (this.TotalElapsedTime > 45)
            {
                this.StartFadeout(5);
            }
        }
        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
        {
            //Override this so we don't take any damage
            return (0);
        }

        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
        {
            base.Collision(oSession, oCollidedWith, nElapsedTime, oCollisionRegion);

            //Give this player the loot!
            if (Globals.Inst().IAmTheServer == true && oCollidedWith.CanPickUpItems == true)
            {
                NetMsg.Send_EntityReceivesLoot(Globals.Inst().GameEngine.DirectPlay,
                                                oCollidedWith.OwnerSocketID, oSession, m_oItemToBeReceived);
                oSession.RemoveEntity(this);
            }

            return (false);
        }
        public override ArrayList GetRegions()
        {
            RegionCircle oCircle = null;
            ArrayList oRegions = new ArrayList();

            oCircle = new RegionCircle(base.MostUpToDateLoc.Pos, m_vSize.Y / 2.0f);
            oRegions.Add(oCircle);

            return (oRegions);
        }
        public override void Render(RenderSettings oRenderSettings)
        {
            string sTexture = string.Empty;
            bool[] baState = null;
            System.Drawing.Color oColor = System.Drawing.Color.White;
            Vector2 vUpperLeftScreenPt = Vector2.Empty;
            Vector2 vLowerRightScreenPt = Vector2.Empty;
            Vector2 vScrSize = Vector2.Empty;
            System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


            if (oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor)
            {
                vUpperLeftScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt(
                                        oRenderSettings, new Vector2(base.Pos.X - m_vSize.X / 2f, base.Pos.Y - m_vSize.Y / 2f));
                vLowerRightScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt(
                                        oRenderSettings, new Vector2(base.Pos.X + m_vSize.X / 2f, base.Pos.Y + m_vSize.Y / 2f));
                vScrSize = new Vector2(vLowerRightScreenPt.X - vUpperLeftScreenPt.X,
                                        vLowerRightScreenPt.Y - vUpperLeftScreenPt.Y);

                oRenderRect = new System.Drawing.Rectangle(
                    (int)vUpperLeftScreenPt.X, (int)vUpperLeftScreenPt.Y,
                    (int)vScrSize.X, (int)vScrSize.Y);

                if (m_oItemToBeReceived.ItemClass == enumItemClass.Routine)
                {
                    oColor = System.Drawing.Color.White;
                    sTexture = "FloatingLoot_Normal";
                }
                else if (m_oItemToBeReceived.ItemClass == enumItemClass.Charged)
                {
                    oColor = System.Drawing.Color.FromArgb(216, 117, 0);
                    sTexture = "FloatingLoot_Other";
                }
                else if (m_oItemToBeReceived.ItemClass == enumItemClass.SuperCharged)
                {
                    oColor = System.Drawing.Color.LightBlue;//Silver;
                    sTexture = "FloatingLoot_Other";
                }
                else if (m_oItemToBeReceived.ItemClass == enumItemClass.Legacy)
                {
                    oColor = System.Drawing.Color.Gold;
                    sTexture = "FloatingLoot_Other";
                }

                Globals.Inst().GameEngine.RenderTexture2D(sTexture,
                    System.Drawing.Rectangle.Empty, oRenderRect,
                    new Vector2(vScrSize.X / 2f, vScrSize.Y / 2f), base.Angle + Math.PI / 2.0,
                    DSMisc.Max(oRenderSettings.PercentTransparent, this.FadeoutTransparency()),
                    false, oColor.ToArgb());

                //If they are holding down Alt, show the contents
                Globals.Inst().GameEngine.DirectInput.GetKeyboardState(ref baState);
                if (baState[(int)Microsoft.DirectX.DirectInput.Key.LeftControl] == true ||
                    baState[(int)Microsoft.DirectX.DirectInput.Key.RightControl] == true)
                {
                    Globals.Inst().GameEngine.RenderText(
                                    oRenderRect.Left - (m_oItemToBeReceived.Name.Length / 2.0) * 2.5, oRenderRect.Top + 10,
                                    oColor, m_oItemToBeReceived.Name);
                }
            }
        }

        public override DSSerialize Serialize(Session oSession)
        {
            DSSerialize oSerialize = new DSSerialize();


            //Now save our data
            oSerialize.Set(0, base.Serialize(oSession));
            oSerialize.Set(1, m_oItemToBeReceived.GetType().Name);
            oSerialize.Set(2, m_oItemToBeReceived.Serialize(oSession));


            return (oSerialize);
        }
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
        {
            Assembly oControlAssembly = null;
            Type oControlType = null;
            object[] oArgs = new object[] { };
            string sTypeName = "";


            base.DeSerialize(oSession, (DSSerialize)oSerialize.Get(0));
            sTypeName = oSerialize.GetString(1);
            try
            {
                //Make our item
                oControlAssembly = Assembly.Load("StellarLanes");//.LoadWithPartialName( "StellarLanes" );
                oControlType = oControlAssembly.GetType("DarkStride.StellarLanes.SharedDLL." + sTypeName);

                //Make it!
                if (oControlType.BaseType == typeof(Chassis) ||
                    oControlType.BaseType == typeof(Module) ||
                    oControlType.BaseType.BaseType == typeof(Module))
                {
                    //Some types are created uniqely
                    oArgs = new object[] { null };
                }
                m_oItemToBeReceived = (Item)Activator.CreateInstance(oControlType, oArgs);
                //Now deserialize the form
                m_oItemToBeReceived.DeSerialize(oSession, (DSSerialize)oSerialize.Get(2));
            }
            catch (System.Exception oEx)
            {

                DSMisc.ShowErrors(oEx);
            }
        }


        #region Properties
        public Item ItemToBeReceived
        {
            get
            {
                return (m_oItemToBeReceived);
            }
            set
            {
                m_oItemToBeReceived = value;
            }
        }
        public override string Name
        {
            get
            {
                return (m_oItemToBeReceived.Name);
            }
        }
        #endregion
    }

    public class SuperMass : Entity
    {
        #region Member Variables
        private double m_nMinRadius = 0;
        #endregion

        public override DSSerialize Serialize(Session oSession)
        {
            DSSerialize oSerialize = new DSSerialize();

            oSerialize.Set(0, base.Serialize(oSession));
            oSerialize.Set(1, m_nMinRadius);

            return (oSerialize);
        }
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
        {
            base.DeSerialize(oSession, (DSSerialize)oSerialize.Get(0));
            m_nMinRadius = oSerialize.GetDouble(1);
        }

        #region Properties
        public double MinRadius
        {
            get
            {
                return (m_nMinRadius);
            }
            set
            {
                m_nMinRadius = value;
            }
        }
        #endregion
    }
    public class Planet : SuperMass
	{
		#region Properties
		private Vector2 m_vSize = new Vector2( 550,550 );
		#endregion

		public Planet()
		{
			this.Mass = 500;
            this.MinRadius = 500;
			this.EntityType = enumEntityType.Astrological; 
		}
        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
		{
			return( 0 );
		}
		public override double DelayBetweenLiteSends()
		{
			return( 30 );
		}

		public override void Render( RenderSettings oRenderSettings )
		{
			string sTexture = "";
			int nFrame = 0;
			Vector2 vScreenPt = Vector2.Empty;
			Vector2 vWorldUpperLeftCorner = Vector2.Empty;
			System.Drawing.Rectangle oRenderRect = System.Drawing.Rectangle.Empty;


			if( oRenderSettings.RenderType == enumRenderType.Topside && oRenderSettings.InGameScreen != enumGameScreen.Editor )
			{
				vWorldUpperLeftCorner = new Vector2( (float)( this.Pos.X - m_vSize.X / 2.0 ),
					(float)( this.Pos.Y + m_vSize.Y / 2.0 ) );
				vScreenPt = Globals.Inst().Session.ConvertWorldPtToScreenPt( oRenderSettings,vWorldUpperLeftCorner );

				nFrame = (int)( Globals.Inst().GameEngine.AppTime % 22.0 );
				sTexture = "Earth_" + nFrame.ToString();

				oRenderRect = new System.Drawing.Rectangle( 
					(int)vScreenPt.X,(int)vScreenPt.Y,
					(int)( m_vSize.X * oRenderSettings.ZoomLevel ),(int)( m_vSize.Y * oRenderSettings.ZoomLevel ) );

				Globals.Inst().GameEngine.RenderTexture2D( sTexture,System.Drawing.Rectangle.Empty,
					oRenderRect,
					Vector2.Empty,0,oRenderSettings.PercentTransparent,false,
					oRenderSettings.BaseDrawColor.ToArgb() );
			}
		}
		public override ArrayList GetRegions()
		{
			ArrayList oRegions = new ArrayList();

			RegionCircle oCircle = new RegionCircle( new Vector2( base.MostUpToDateLoc.Pos.X,base.MostUpToDateLoc.Pos.Y ),m_vSize.X / 2.0f );
			oRegions.Add( oCircle );

			return( oRegions );
		}
	}
    public class Portal : Entity
    {
        #region Member Variables
        private const double m_cMINEJECTVEL = 800;
        private const double m_cTIMETILL_SOURCEFLARE = 3.0;
        private const double m_cTIMETILL_TRANSITION = .5;
        private const double m_cTIMETILL_MOTIONRESUMES = .5;
        private const double m_cTIMETILL_TOTALTIME = m_cTIMETILL_SOURCEFLARE + m_cTIMETILL_TRANSITION + m_cTIMETILL_MOTIONRESUMES;

        private SLPortalEffect m_oPortalEffect = null;
        private Vector2 m_vSize = new Vector2(10, 10);
        private DSNetPKey m_oLinkedPortalEntityID = null;

        private bool m_bShowingActive = true;
        private System.Collections.Generic.SortedList<string, RecentPortalUsers> m_oRecentUsers = new System.Collections.Generic.SortedList<string, RecentPortalUsers>();
        private System.Collections.Generic.List<TransitionInProgress> m_oUsersInTransit = new System.Collections.Generic.List<TransitionInProgress>();
        #endregion


        public Portal()
        {
            this.Mass = 5000;
            //this.MinRadius = 750;
            this.EntityType = enumEntityType.Astrological;
        }

        public override void Advance(Session oSession, double nElapsedTime)
        {
            TransitionInProgress oLoopTransit = null;
            bool bContains = true;


            base.Advance(oSession, nElapsedTime);

            //Advance our timers
            AdvanceRecentUsers(nElapsedTime);

            #region Update our transition of the ship
            for (int i = 0; i < m_oUsersInTransit.Count; i++)
            {
                oLoopTransit = (TransitionInProgress)m_oUsersInTransit[i];
                if (i >= m_oUsersInTransit.Count) { break; }

                if (oLoopTransit.m_nTimeTillEntityTransmits > 0)
                {
                    oLoopTransit.m_nTimeTillEntityTransmits -= nElapsedTime;
                    oLoopTransit.m_oEntityTransmitting.Vel = new Vector2(0, 0);

                    //Launch source flare
                    if (oLoopTransit.m_nTimeTillEntityTransmits < m_cTIMETILL_TOTALTIME - m_cTIMETILL_SOURCEFLARE && 
                        oLoopTransit.m_bHaveLaunched_SourceFlare == false)
                    {
                        oLoopTransit.m_bHaveLaunched_SourceFlare = true;
                        if (Globals.Inst().IAmTheServer == false)
                        {
                            SLPortalEffectSourceTransmit.StartPortalSourceTransmitEffect(oSession, this.Pos, this.Location.ZoneID.ToString());
                        }
                    }
                    //Transmit the entity
                    else if (oLoopTransit.m_nTimeTillEntityTransmits < m_cTIMETILL_TOTALTIME - (m_cTIMETILL_SOURCEFLARE + m_cTIMETILL_TRANSITION) && 
                             oLoopTransit.m_bHaveLaunched_Transmit == false)
                    {
                        Portal oPartnerPortal = GetPartnerPortal(oSession);

                        oLoopTransit.m_bHaveLaunched_Transmit = true;

                        oLoopTransit.m_oEntityTransmitting.Location.ZoneID = oPartnerPortal.Location.ZoneID;
                        oLoopTransit.m_oEntityTransmitting.Location.Pos = oPartnerPortal.Location.Pos;

                        oSession.TransmitEntity(oLoopTransit.m_oEntityTransmitting, false);
                        oSession.ChangeZone(oLoopTransit.m_oEntityTransmitting);
                        oSession.TransmitEntity(this, true);
                        oSession.TransmitEntity(oPartnerPortal, true);

                        //Speed up the departing velocity
                        if (oLoopTransit.m_oInitialVel.Length() == 0)
                        {
                            double nAngle = DSMisc.GetRnd() * Math.PI * 2.0;
                            oLoopTransit.m_oInitialVel = new Vector2((float)(Math.Cos(nAngle) * m_cMINEJECTVEL), (float)(Math.Sin(nAngle) * m_cMINEJECTVEL));
                        }
                        else if (oLoopTransit.m_oInitialVel.Length() < m_cMINEJECTVEL)
                        {
                            oLoopTransit.m_oInitialVel *= (float)(m_cMINEJECTVEL / oLoopTransit.m_oInitialVel.Length());
                        }

                        if (Globals.Inst().IAmTheServer == false)
                        {
                            SLPortalEffect.StartPortalDestTransmitEffect(oSession, oPartnerPortal.Pos, 
                                        oLoopTransit.m_oInitialVel, oPartnerPortal.Location.ZoneID.ToString());
                        }
                    }
                    //we have to do this AFTER we transmit the entity.  This is because we measure the ships
                    //location, transmit them, then update the stars.  Since they are jumping in space the stars
                    //get all screwed up.  So we have to transmit the ship, wait one frame, then randomize the stars.
                    else if (oLoopTransit.m_nTimeTillEntityTransmits < m_cTIMETILL_TOTALTIME - (m_cTIMETILL_SOURCEFLARE + m_cTIMETILL_TRANSITION) &&
                             oLoopTransit.m_bHaveLaunched_StarsRandomized == false)
                    {
                        oLoopTransit.m_bHaveLaunched_StarsRandomized = true;

                        if (oSession.Starscape != null && oLoopTransit.m_oEntityTransmitting == Globals.Inst().OurShip )
                        {
                            oSession.Starscape.RandomizeStars();
                        }
                    }
                    //Launch them from the portal on the other side
                    else if (oLoopTransit.m_nTimeTillEntityTransmits <= 0 && oLoopTransit.m_bHaveLaunched_Motion == false)
                    {
                        oLoopTransit.m_bHaveLaunched_Motion = true;
                        oLoopTransit.m_oEntityTransmitting.Vel = oLoopTransit.m_oInitialVel;
                    }

                    //If we haven't launched them yet, then keep their velocity 0
                    if (oLoopTransit.m_bHaveLaunched_Motion == false)
                    {
                        oLoopTransit.m_oEntityTransmitting.Vel = new Vector2(0, 0);
                    }
                }
                else
                {
                    m_oUsersInTransit.RemoveAt(i);
                    i--;
                }
            }
            #endregion

            if (Globals.Inst().IAmTheServer == false)
            {
                #region Update our portal effect
                if (Globals.Inst().MyZone.ZoneID == this.Location.ZoneID &&
                    m_oPortalEffect == null)
                {
                    m_oPortalEffect = SLPortalEffect.StartPortalEffect(oSession, this.Location.Pos,this.Location.ZoneID.ToString(),true);
                    m_bShowingActive = true;
                    m_oPortalEffect.Advance(.25);                    
                }
                else if (Globals.Inst().MyZone.ZoneID != this.Location.ZoneID &&
                         m_oPortalEffect != null)
                {
                    Globals.Inst().GameEngine.ParticleSystem.RemoveParticleManager(m_oPortalEffect.GUID);
                    m_oPortalEffect = null;
                }
                #endregion

                //Update the portal with our current usability state
                if (m_oPortalEffect != null && Globals.Inst().IAmTheServer == false && Globals.Inst().OurShip != null)
                {
                    bContains = m_oRecentUsers.ContainsKey(Globals.Inst().OurShip.PKey);
                    if (m_bShowingActive == true && bContains == true)
                    {
                        //Switch to in-active
                        m_bShowingActive = false;
                        m_oPortalEffect.RunDown = true;
                        m_oPortalEffect = SLPortalEffect.StartPortalEffect(oSession, this.Location.Pos, this.Location.ZoneID.ToString(),false);
                    }
                    else if (m_bShowingActive == false && bContains == false )
                    {
                        //Switch to active
                        m_bShowingActive = true;
                        m_oPortalEffect.RunDown = true;
                        m_oPortalEffect = SLPortalEffect.StartPortalEffect(oSession, this.Location.Pos, this.Location.ZoneID.ToString(), true);
                    }
                }
            }
        }
        public override void Render(RenderSettings oRenderSettings)
        {
            Portal oPartnerPortal = null;
            Zone oParnerZone = null;
            Vector2 vScrPos = Vector2.Empty;
            double nHUDStartY = 0;
            string sName = string.Empty;


            base.Render(oRenderSettings);

            oPartnerPortal = GetPartnerPortal(Globals.Inst().Session);
            if (oPartnerPortal != null)
            {
                oParnerZone = Globals.Inst().Session.Zones[oPartnerPortal.Location.ZoneID.ToString()];
                sName = "Portal to: " + oParnerZone.Name;

                vScrPos = Globals.Inst().Session.ConvertWorldPtToScreenPt(oRenderSettings, this.Pos);
                nHUDStartY = (int)(vScrPos.Y + 140 * oRenderSettings.ZoomLevel);

                Globals.Inst().GameEngine.RenderText(
                        vScrPos.X - (sName.Length / 1.0) * 2.5, nHUDStartY - 7,
                        System.Drawing.Color.White, sName);
            }
            else
            {
                sName = "Woa, error here.";
            }
        }
        public override ArrayList GetRegions()
        {
            ArrayList oRegions = new ArrayList();

            RegionCircle oCircle = new RegionCircle(new Vector2(base.MostUpToDateLoc.Pos.X, base.MostUpToDateLoc.Pos.Y), m_vSize.X / 2.0f);
            oRegions.Add(oCircle);

            return (oRegions);
        }
        public override bool Collision(Session oSession, Entity oCollidedWith, double nElapsedTime, RegionPoint oCollisionRegion)
        {
            Portal oPartnerPortal = GetPartnerPortal(oSession);
            TransitionInProgress oTransit = null;


            if (oPartnerPortal != null && oCollidedWith.GetType() != typeof( Portal ) &&
                oCollidedWith.OwnerSocketID != MiscConstants.m_cNOSOCKETIDASSIGNED &&
                m_oRecentUsers.ContainsKey(oCollidedWith.PKey.ToString()) == false)
            {
                Globals.PlaySound(SoundConstants.m_cPORTALTRANSIT,this.Location);

                //Don't allow this guy to transition again for 15 seconds
                oPartnerPortal.AddRecentUser(oCollidedWith);
                this.AddRecentUser(oCollidedWith);

                //Setup our graphical displays of the portal transition
                oTransit = new TransitionInProgress();
                oTransit.m_oInitialVel = oCollidedWith.Vel;
                oTransit.m_nTimeTillEntityTransmits = m_cTIMETILL_TOTALTIME;
                oTransit.m_oEntityTransmitting = oCollidedWith;
                m_oUsersInTransit.Add(oTransit);

                //Stop our guy
                oCollidedWith.Vel = new Vector2(0, 0);
                oCollidedWith.Pos = this.Pos;
                oSession.TransmitEntity(oCollidedWith, false);

                oCollidedWith.StartShrinkEeffect(m_cTIMETILL_SOURCEFLARE, m_cTIMETILL_TRANSITION + m_cTIMETILL_MOTIONRESUMES);
            }            


            return(false);
        }
        private Portal GetPartnerPortal(Session oSession)
        {
            Portal oPartnerPortal = null;


            if (m_oLinkedPortalEntityID != null)
            {
                oPartnerPortal = (Portal)oSession.GetEntity(m_oLinkedPortalEntityID.ToString());
            }


            return (oPartnerPortal);
        }
        public override double DealDamage(Session oSession, Entity oEntityDealingDamage, double nAmount, Region oDamageWhere)
        {
            return (nAmount);
        }

        public void AddRecentUser(Entity oEntityWhoUsedMe)
        {
            RecentPortalUsers oRecent = null;

            if (m_oRecentUsers.ContainsKey(oEntityWhoUsedMe.PKey) == false)
            {
                m_oRecentUsers.Add(oEntityWhoUsedMe.PKey, new RecentPortalUsers(oEntityWhoUsedMe));
            }
            else
            {
                oRecent = m_oRecentUsers[oEntityWhoUsedMe.PKey];
                oRecent.m_nTimeSinceLastUse = 0;
            }
        }
        private void AdvanceRecentUsers(double nElapsedTime)
        {
            RecentPortalUsers oLoopRecent = null;

            for (int i = 0; i < m_oRecentUsers.Count; i++)
            {
                if (i >= m_oRecentUsers.Count) { break; }
                oLoopRecent = m_oRecentUsers.Values[i];

                oLoopRecent.m_nTimeSinceLastUse += nElapsedTime;
                if (oLoopRecent.m_nTimeSinceLastUse > 15)
                {
                    m_oRecentUsers.RemoveAt(i);
                    i--;
                }
            }
        }

        public override DSSerialize Serialize(Session oSession)
        {
            DSSerialize oSerialize = new DSSerialize();

            oSerialize.Set(0, base.Serialize(oSession));
            if (m_oLinkedPortalEntityID != null)
            {
                oSerialize.Set(1, m_oLinkedPortalEntityID.Serialize());
            }

            return (oSerialize);
        }
        public override void DeSerialize(Session oSession, DSSerialize oSerialize)
        {
            base.DeSerialize(oSession, (DSSerialize)oSerialize.Get(0));

            if (oSerialize.Get(1) != null)
            {
                m_oLinkedPortalEntityID = new DSNetPKey(-1, -1);
                m_oLinkedPortalEntityID.DeSerialize((DSSerialize)oSerialize.Get(1));
            }
            else
            {
                m_oLinkedPortalEntityID = null;
            }

        }

        internal class RecentPortalUsers
        {
            public double m_nTimeSinceLastUse = 0;
            public Entity m_oEntity = null;

            public RecentPortalUsers(Entity oEntity)
            {
                m_oEntity = oEntity;
            }
        }
        internal class TransitionInProgress
        {
            public Entity m_oEntityTransmitting = null;
            public double m_nTimeIntoTransit = 0;
            public Vector2 m_oInitialVel = new Vector2(0, 0);
            public double m_nTimeTillEntityTransmits = 0;

            public bool m_bHaveLaunched_SourceFlare = false;
            public bool m_bHaveLaunched_Transmit = false;
            public bool m_bHaveLaunched_StarsRandomized = false;
            public bool m_bHaveLaunched_Motion = false;
        }


        #region Properties 
        public DSNetPKey LinkedPortalEntityID
        {
            get
            {
                return (m_oLinkedPortalEntityID);
            }
            set
            {
                m_oLinkedPortalEntityID = value;
            }
        }
        #endregion
    }    
}
